function [colo_data,punc_data] = colo_info(varargin)
%This function calculates the colocalization information from two vertices
%list.  It will calculate the sigmoid and the slopes of the colocalization.
%Synatax:   [colo_data,punc_data] = colo_info('sdist',20,'z',0.7); 
%Input:     This function does not really require any inputs.  However if
%           you do not want default values then.
%           'sdist' = the search distance.  Default = 10 pixels
%           'vert1' = the first set of vertices
%           'vert2' = the second set of vertices
%           'x' = the scale, as a ratio or xyz, of 1 unit in the x axis.
%                 Default = 1
%           'y' = the scale, as a ratio or xyz, of 1 unit in the y axis.
%                 Default = 1
%           'z' = the scale, as a ratio or xyz, of 1 unit in the z axis.
%                 Default = 1
%           't_range' = the distance range you want to map out a three number
%                 value [a b c].  a = min, b = max, c = increment.
%                 Default = [0 15 0.1].  So 150 steps.
%           'fig' = Turns figure creation on and off.  Default = 1(on)
%           'sav' = save the colo data.  Default = 1(on)
%           'multi' = turn on multi thread processing.  Default = 0(off)
%           'rnd' = turn on calculate random permutations. Default = 0(off)
%Output:    colo_data = a structure that holds the colo information, it has
%               these fields:
%               'colo_trend' = the number of colocalization events as a
%                   function of each colocalization distance threshold
%               'slope' = the slope at each point of data
%               'mean' = the mean distance between the two vertices list
%               'median' = the median of the distances
%               'std' = the standard deviation of the distances.
%               'thresh_dist' = the threshold distance of each point.
%           punc_data = Good old punc_data structure, if you just want to
%               poke around.


[sdist,vert1,vert2,x_scale,y_scale,z_scale,t_range,fig,sav,multi,rnd] = parse(varargin);     %Parse the input

%lets open up the files
filenames = {};

%get files. Note: currently the filterindex is the filter selected for the file selection.
if ~vert1    %if the user did not enter a first vertices list
    %generate punc_data for the pair
    prompt_box('title','Open vertex list One','prompt1','For Colocolization','position','center');
    pause(0.25);
    [filename_tmp,pathname,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.xlsx','Excel 2007 files (*.xlsx)';...
        '*.xls','Excel files (*.xls)';'*.mat','Mat files (*.mat)';'*.*','All Files';},...
        'Open First Vertices File','Multiselect','off');
    %Now open the data sets according to the filter used.
    switch filterindex
        case {2,3}              %excel file
            vertices{1} = single(xlsread([pathname,filename_tmp]));
        case {1,5}                  %text file - Default coma delimited
            vertices{1} = single(dlmread([pathname,filename_tmp],',',1,0));
        otherwise               %MAT FILES      - Note Not supported right now. - why do I have it...eh!
            error('Currently only .xls and .csv files are supported');
    end
else    %user entered vertices 1
    vertices{1} = vert1;
    filename_tmp = 'vert1';
    if sav
        pathname = uigetdir2('','Where do you want to save the Information?');       %get the directory where you want to save into.
        %store the file properties (Here because of dependencies)
        punc_data(1).pathnames = pathname;
    end
end
filenames = cat(2,filenames,filename_tmp);

%second file
if ~vert2    %if the user did nto enter a second vert list
    prompt_box('title','Open vertex list Two','prompt1','For Colocolization','position','center');
    pause(0.25);
    [filename_tmp,pathname_tmp] = uigetfile2({'*.csv','Text files (*.csv)';'*.xlsx','Excel 2007 files (*.xlsx)';...
        '*.xls','Excel files (*.xls)';'*.mat','Mat files (*.mat)';'*.*','All Files';},...
        'Open Second Vertices File','Multiselect','off');
    %Now open the data sets according to the filter used.
    switch filterindex
        case {2,3}              %excel file
            vertices{2} = single(xlsread([pathname_tmp,filename_tmp]));
        case {1,5}                  %text file - Default coma delimited
            vertices{2} = single(dlmread([pathname_tmp,filename_tmp],',',1,0));
        otherwise               %MAT FILES      - Note Not supported right now. - why do I have it...eh!
            error('Currently only .xls and .csv files are supported');
    end
else    %user entered vertices 1
    vertices{2} = vert2;
    filename_tmp = 'vert2';
end
filenames = cat(2,filenames,filename_tmp);
punc_data(1).filenames = filenames;     %compatability issues

%now calculate the distances
if multi
    parpool
end
[punc_data,all_data] = distance_cal(filenames,vertices,punc_data,sdist,x_scale,y_scale,z_scale);
if multi
    delete(gcp('nocreate'))
end

%lets generate a couple of interesting numbers
dist = punc_data(1).distance(:,:,2);
dist(isnan(dist(:,1)),:) = [];
colo_data(1).mean = mean(dist);
colo_data(1).median = median(dist);
colo_data(1).std = std(dist);
try
    colo_data(1).imgsize = single(dlmread([pathname,'img_data.csv'],',',1,0));
catch   %ok fail go to default
    colo_data(1).imgsize = [max(vertices{1}(:,1)) max(vertices{1}(:,2)) max(vertices{1}(:,3))];
end

%now generate the colocalization dataset
idx = 0;    %to keep track because i might not be integers
for i = t_range(1):t_range(3):t_range(2)  %go through the range of desired values
    idx = idx+1;    %step
    [class] = punc_colo(punc_data,1,2,'thresh',i);  %generate the colocalization
    colo_number(idx,1) = class(1).total_number;     %grab the number of colocalizations
    thresh_dist(idx,1) = i;     %store the distances ticks
end
colo_data(1).colo_trend = colo_number;  %store the colocalization trend
colo_data(1).thresh_dist = thresh_dist;

%now calculate the slope at each point
slope = 0;  %initiate
for j = 2:size(colo_number,1)   %step through the numbers
    slope(j,1) = (colo_number(j-1)-colo_number(j))/(thresh_dist(j-1)-thresh_dist(j));   %equation of the slope
end
colo_data(1).slope = slope;

%now plot the two
if fig  %figure on or off
    h1 = figure;
    plot(colo_number);
    title('Colocalization as a Function of Threshold Distance');
    ylabel('# of Colocalization Events');
    xlabel(['Threshold Distance x',num2str(t_range(3))]);
    %slope figure
    h2 = figure;
    plot(slope);
    title('Slope of Change as a Function of Threshold Distance');
    ylabel('Slope')
    xlabel(['Threshold Distance x',num2str(t_range(3))]);
end
%save out
if sav
    label = {'ThresholdDistance','ColocalizationEvents','slope'};
    data_tmp = dataset({[thresh_dist colo_number slope],label{:}});
    sav2csv(data_tmp,['coloinfo_',filenames{1}(1:end-4),'_vs_',filenames{2}(1:end-4)],pathname);
end


%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [sdist,vert1,vert2,x_scale,y_scale,z_scale,t_range,fig,sav,multi,rnd] = parse(input)

sdist = 10;  %Default Initialized.
x_scale = 1;    %base scale of a unit in the x axis
y_scale = 1;    %base scale of a unit in the y axis
z_scale = 1;    %base scale of a unit in the z axis
vert1 = 0;
vert2 = 0;
t_range = [0 15 0.1];
fig = 1;
sav = 1;
multi = 1;
rnd = 0;

%Parse the input
if ~isempty(input)
    for i = 1:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'sdist'
                    sdist = input{1,i+1};
                case 'vert1'
                    vert1 = input{1,i+1};
                case 'vert2'
                    vert2 = input{1,i+1};
                case 'x'
                    x_scale = input{1,i+1};
                case 'y'
                    y_scale = input{1,i+1};
                case 'z'
                    z_scale = input{1,i+1};
                case 't_range'
                    t_range = input{1,i+1};
                case 'fig'
                    fig = input{1,i+1};
                case 'sav'
                    sav = input{1,i+1};
                case 'multi'
                    multi = input{1,i+1};
                case 'rnd'
                    rnd = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        end
    end
end

%--------------------------------------------------------------------------
%subfunction to calculate the distance metric.
function [punc_data,all_data] = distance_cal(filenames,vertices,punc_data,nhood,x_scale,y_scale,z_scale)

%now lets calculate the distance metric
%h = waitbar(0,['Data Sets Analyzed: ',filenames{1}],'position',[20 300 275 50]);    %initialize progress bar.
for i = 1:size(filenames,2)      %calculate for each data set
    %seperate out the brightness value
    verti_tmp = vertices{1,i}(:,1:3);     %pull out one data set to work with
    bright_tmp = single(vertices{1,i}(:,4:end));        %brightness data for each vertices
    verti_tmp = single(verti_tmp.*repmat([x_scale y_scale z_scale],size(verti_tmp,1),1));   %scale the vertices to the proper unit
    %preload cell arrays
    dist_tmp = zeros(size(verti_tmp,1),1,size(filenames,2));       %shortest distances
    %verti_match = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching original verticies for the closeset points.
    %v_match_lum = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching luminance values
    %dist_verti_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %the matching distance matrix to the closest point list. A cell array as well.
    %closest_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the closest pivot.  A cell array.
    %closest_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the matching lumninace data.  A cell array
    termi_tmp = zeros(size(verti_tmp,1),3,size(filenames,2));       %the termi vertices
    termi_prop_tmp = zeros(size(verti_tmp,1),size(bright_tmp,2),size(filenames,2)); %the termi properties
    %short_tmp = cell(size(verti_tmp,1),1,size(filenames,2));    %The short list.  A cell Array.
    %short_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));    %The short list luminance.  A cell array.
    %sl_dist_tmp = cell(size(verti_tmp,1),1,size(filenames,2));     %The short list distances.  A cell Array.
    %sl_verti_match_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching verticies for the short list points.
    %sl_v_match_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching luminance data.
    %start progress bar
    %h2 = waitbar(0,['Comparing Data Set: ',filenames{1}],'position',[20 200 275 50]);    %initialize progress bar.
    for j = 1:size(filenames,2)      %get the comparision set
        if j~=i                     %skip if it is the base set
            verti_co_tmp = vertices{1,j}(:,1:3).*repmat([x_scale y_scale z_scale],size(vertices{1,j},1),1);   %scale the vertices to the proper unit
            bright_co_tmp = vertices{1,j}(:,4:end);     %matching brightness data for comparision channel, now include other properties
            %h3 = waitbar(0,'Vertex Distance Analyzed: 0');    %initialize progress bar.
            parfor k = 1:size(verti_tmp,1)     %iterate through each vertex of the set
                display(['distance ',filenames{1,i},' ',filenames{1,j},' ',num2str(i),'_',num2str(j),'_',num2str(k)]);
                [short_list,short_list_lum] = find_neighbors(verti_tmp(k,:),verti_co_tmp,nhood*2,bright_co_tmp);    %get a short list of local pivots (removed:,bright_co_tmp)
                curr_verti = repmat(verti_tmp(k,:),size(short_list,1),1);         %replicate the current vertex to the size of the short list.
                %now we are going to make this a search in a sphere
                verti_dist = single(dddist(curr_verti,short_list));     %calculate the distance of the current vertex from all points in the short list.
                idx = verti_dist>nhood;     %any point actually outside of the specified radius
                verti_dist(idx) = [];       %gone
                short_list(idx,:) = [];     %gone
                short_list_lum(idx,:) = []; %gone
                curr_verti(idx,:) = [];     %gone
                if ~isempty(short_list)           %make sure there is something in the neighborhood.
                    [x,y] = find(verti_dist==absmin(verti_dist));   %find the position of the closest point
                    %closest_tmp{k,1,j} = short_list(x,:);           %create a cell array of all the closest points, just in case of multiples.
                    %closest_lum_tmp{k,1,j} = short_list_lum(x,:);   %create a matching cell array of luminance
                    termi_tmp(k,:,j) = short_list(x(1,1),:);        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                    termi_prop_tmp(k,:,j) = short_list_lum(x(1,1),:);  %termi properties to match termi
                    %verti_match{k,1,j} = repmat(verti_tmp(k,:),size(short_list(x,:),1),1);    %create a matching original verticies for the closets pts.
                    %v_match_lum{k,1,j} = repmat(bright_tmp(k,:),size(short_list(x,:),1),1);
                    %dist_verti_tmp{k,1,j} = verti_dist(x,:);  %create a matching set of distances with the vertices.
                    dist_tmp(k,1,j) = verti_dist(x(1,1),:);              %create a columnar vector of shortest distances
                    vert_prop(k,:,j) = bright_tmp(k,:);         %store the vertices associated properties
                    short_tmp{k,1,j} = single(short_list);    %output the short list as well.
                    short_lum_tmp{k,1,j} = single(short_list_lum);  %output the luminances of the short list as well
                    sl_dist_tmp{k,1,j} = single(verti_dist);  %output the distances for all points in the short list.
                    all_verti{k,1,j} = single([curr_verti repmat(k,size(curr_verti,1),1)]);    %create a matching original verticies for the all near pts & put in the ordinals
                    all_prop{k,1,j} = single(repmat(bright_tmp(k,:),size(curr_verti,1),1));     %create a matching properties array.
                    %sl_verti_match_tmp{k,1,j} = repmat(verti_tmp(k,:),size(short_list,1),1);  %create a matching vertices for the short list
                    %sl_v_match_lum_tmp{k,1,j} = repmat(bright_tmp(k,:),size(short_list,1),1);  %create a matching luminance for the short list
                else                        %there is no one around me (sad)
                    %closest_tmp{k,1,j} = single([NaN NaN NaN]);           %not a number for easy extraction.
                    %closest_lum_tmp{k,1,j} = single(NaN(1,size(bright_tmp,2)));
                    termi_tmp(k,:,j) = single([NaN NaN NaN]);
                    termi_prop_tmp(k,:,j) = single(NaN(1,size(bright_tmp,2)));
                    %verti_match{k,1,j} = verti_tmp(k,:);    
                    %v_match_lum{k,1,j} = bright_tmp(k,:);
                    %dist_verti_tmp{k,1,j} = single(NaN);  
                    dist_tmp(k,1,j) = single(NaN);
                    vert_prop(k,:,j) = single(NaN(1,size(bright_tmp,2)));
                    short_tmp{k,1,j} = single([NaN NaN NaN]);    %output the short list as well.
                    short_lum_tmp{k,1,j} = single(NaN(1,size(bright_tmp,2)));
                    sl_dist_tmp{k,1,j} = single(NaN);   %output the distance for short list.
                    all_verti{k,1,j} = single([verti_tmp(k,:) k]);
                    all_prop{k,1,j} = single(bright_tmp(k,:));
                    %sl_verti_match_tmp{k,1,j} = verti_tmp(k,:);  %all by yourself
                    %sl_v_match_lum_tmp{k,1,j} = bright_tmp(k,:);
                end
                %waitbar(k/size(verti_tmp,1),h3,['Vertex Distance Analyzed: ',num2str(k)]);   %update progress
            end
            %close(h3);   %close progress bar
            %waitbar(j/size(filenames,2),h2,['Comparing Data Set: ',filenames{j}]);   %update progress
        end
    end
    %store the data. The structure is that the distance is a matrix that
    %corresponds with the vertices list in the rows and with each z step a
    %calculation against another dataset, the self to self comparision data
    %set is all 0.  So 4 datasets, will be a matrix of (x,1,4).  The
    %vertices is just the verticies used to calculate the distance.  The
    %closest_pt dataset is a cell array that matches the verticies list in
    %rows and with each z step representing another dataset, the cell array
    %is used for the instances where there are multiple closest points.
    punc_data(i).distance = single(dist_tmp);       %shortest distances
    punc_data(i).vertices = single(horzcat(verti_tmp,linspace(1,size(verti_tmp,1),size(verti_tmp,1))'));      %the vertices + the oridinals
    punc_data(i).vert_prop = single(vert_prop);     %the property value for only the vertices that have a neighbor in that particular channel
    punc_data(i).vert_all_prop = single(bright_tmp);          %The property value for all of the vertices in this channel
    %punc_data(i).verti_match = verti_match; %the matching original verticies for the closeset points.
    %punc_data(i).v_match_lum = v_match_lum; %the matching luminance values
    %punc_data(i).dist_verti = dist_verti_tmp;   %the matching distance matrix to the closest point list. A cell array as well.
    %punc_data(i).closest_pt = closest_tmp;  %the closest pivot.  A cell array.
    %punc_data(i).closest_prop = closest_lum_tmp;  %the matching property data for the closest pivots.  A cell array
    punc_data(i).termi = single(termi_tmp);         %the matching terminal vertices
    punc_data(i).termi_prop = single(termi_prop_tmp);   %the matching properties for the terminal vertices
    all_data(i).all_termi = short_tmp;    %The short list.  A cell Array.
    all_data(i).all_tprop = short_lum_tmp;    %The short list luminance.  A cell array.
    all_data(i).all_dist = sl_dist_tmp;     %The short list distances.  A cell Array.
    all_data(i).all_vert = all_verti;      %a matching ori vertices array to all_termi.  A cell array.
    all_data(i).all_prop = all_prop;       %a matching ori properties array to all_termi.  A cell array.
    %punc_data(i).sl_verti_match = sl_verti_match_tmp;   %The matching verticies for the short list points.
    %punc_data(i).sl_v_match_lum = sl_v_match_lum_tmp;   %The matching luminance data.
    %update progress bars
    %close(h2);   %close progress bar
    %waitbar(i/size(filenames,2),h,['Data Sets Analyzed: ',filenames{i}]);   %update progress
    clear short_lum_tmp sl_dist_tmp all_verti all_prop short_tmp
end
%close(h);   %close progress bar